#include "heuristics.hpp"

void GameNewState::applyAction(ActionHist act) {
	this->state.applyAction(act);
	this->actions.push_back(act);
}
GameNewState::GameNewState(VirtGameState& s) {
	this->state = s;
}

/**
	Heuristics functions
**/

struct AgentChoice {
	double score;
	int agentId, alienId;
};
bool operator < (const AgentChoice& a, const AgentChoice& b) {
	return a.score < b.score;
}
bool operator > (const AgentChoice& a, const AgentChoice& b) {
	return a.score > b.score;
}

double scoreAgentToAlien_h1(int iAgent, alien_info alien, VirtGameState& state, bool forcePos = false) { // Only on aliens without enemies
	position& agentPos = state.myAgents[iAgent].pos;
	if (!state.map.canMyAgentGoTo(alien.pos, iAgent) && !forcePos) {
		return -INF;
	}
	if (alien.capture_en_cours == NB_TOURS_CAPTURE) {
		return -INF;
	}
	if (state.turnId + conf::TOUR_AVANT_IGNORE < alien.tour_invasion) {
		return 0;
	}

	int nbTurnToGo = (getBaseDistTo(agentPos, alien.pos) + NB_POINTS_ACTION-1) / NB_POINTS_ACTION;
	int nbTurnMove = max(nbTurnToGo, alien.tour_invasion - state.turnId);
	int nbTurnCapture = nbTurnMove + NB_TOURS_CAPTURE - alien.capture_en_cours;

	if (alien.tour_invasion + alien.duree_invasion < nbTurnCapture) {
		return -INF;
	}
	double score = alien.points_capture * 100;
	score = score * 3. / (double)nbTurnCapture;
	score -= 0.001 * getBaseDistTo(agentPos, alien.pos);

	if (alien.pos == agentPos) {
		score *= 10;
	}

	score -= score * conf::malus_action_wait_per_turn * max(0, nbTurnMove - nbTurnToGo)

	return score;
}

double scoreAgentToAlien_h2(int iAgent, alien_info alien, VirtGameState& state) { // Only on aliens without enemies
	double score = scoreAgentToAlien_h1(iAgent, alien, state);
	score += (double)(rand()%50) * score / 100.;
	return score;
}

double scoreAgentToAlien_h_agro(int iAgent, alien_info alien, VirtGameState& state) {
	double score = scoreAgentToAlien_h1(iAgent, alien, state, true);
	if (state.map.get(alien.pos).type == CELL_AGENT_ENEMY) {
		int turnsToAttack = (getBaseDistTo(state.myAgents[iAgent].pos, alien.pos) + COUT_POUSSER
				+ NB_POINTS_ACTION-1)/ NB_POINTS_ACTION;
		score = score * max(1., (double)turnsToAttack) / max(1., (double)(turnsToAttack - 2));

		if (turnsToAttack <= NB_TOURS_CAPTURE - alien.capture_en_cours) {

			int nbHard = 0, nbSoft = 0;
			for (int iMove = 0; iMove < 2; iMove++) {
				auto pos1 = alien.pos + moves[iMove];
				auto pos2 = alien.pos + moves[iMove+2];
				if (state.map.isWall(pos1) || state.map.isWall(pos2)) {
					nbHard += 1;
				} else if (state.map.isEnemy(pos1) || state.map.isEnemy(pos2)) {
					nbSoft += 1;
				}
			}
			if (nbHard >= 2) {
				return 0;
			} else if (nbHard == 1) {
				score -= score * ((double)nbSoft * conf::coeffHardDefense);
			} else {
				score -= score * ((double)nbSoft * conf::coeffSoftDefense);
			}

			score += score * conf::coeffAttackBonus;
		} else {
			return 0;
		}
	}
	return score;
}

/**
	Base des heuristiques
**/

double MoveChoice::getScoreAgentToAlien(int iAgent, alien_info alien, int heuristic) { // Only on aliens without enemies
	if (heuristic == HEURISTIC_1) {
		return scoreAgentToAlien_h1(iAgent, alien, this->state);
	} else if (heuristic == HEURISTIC_2) {
		return scoreAgentToAlien_h2(iAgent, alien, this->state);
	} else if (heuristic == HEURISTIC_AGRO) {
		return scoreAgentToAlien_h_agro(iAgent, alien, this->state);
	} else {
		cerr << "ERRROR !";
	}
	return 0;
}

MoveChoice::MoveChoice(VirtGameState& s) : state(s) {}

target_array MoveChoice::getObjectives(int heuristic) {
	//cerr << "\n===== Make choices " << heuristic << "\n";
	array<int, NB_AGENTS> agentDest = {-1, -1, -1, -1};
	vector<AgentChoice> choices;

	for (int iAgent = 0; iAgent < NB_AGENTS; iAgent++) {
		for (int iAlien = 0; iAlien < this->state.nbAliens; iAlien++) {
			double score = this->getScoreAgentToAlien(iAgent, this->state.aliens[iAlien], heuristic);
			choices.push_back({score, iAgent, iAlien});
		}
	}
	sort(choices.begin(), choices.end(), greater<AgentChoice>());
	for (AgentChoice& choose : choices) {
		if (agentDest[choose.agentId] == -1) {
			bool isAttack = this->state.map.isEnemy(this->state.aliens[choose.alienId].pos);
			bool defended = false;
			int n = count(agentDest.begin(), agentDest.end(), choose.alienId);
			if (isAttack) {
				for (int iMove = 0; iMove < 4; iMove++) {
					if (this->state.map.isEnemy(this->state.aliens[choose.alienId].pos + moves[iMove])) {
						defended = true;
					}
				}
			}

			if (n == 0 || (n < conf::MAX_ON_ATTACK && isAttack && defended)) {
				agentDest[choose.agentId] = choose.alienId;
				debug_afficher_drapeau(this->state.aliens[choose.alienId].pos, DRAPEAU_ROUGE);
				/*cerr << "Agent " << this->state.myAgents[choose.agentId].pos.ligne
					<< " " << this->state.myAgents[choose.agentId].pos.colonne
					 << " go to " << this->state.aliens[choose.alienId].pos.ligne
					 << " " << this->state.aliens[choose.alienId].pos.colonne << "\n";*/
			}
		}
	}

	target_array result;
	for (int iAgent = 0; iAgent < NB_AGENTS; iAgent++) {
		if (agentDest[iAgent] == -1) {
			result[iAgent] = {this->getAgentSimpleDest(iAgent), TARGET_POS};
		} else {
			int targetType = TARGET_ALIEN;
			if (this->state.map.isEnemy(this->state.aliens[agentDest[iAgent]].pos)) {
				targetType = TARGET_ATTACK;
			}
			result[iAgent] = {this->state.aliens[agentDest[iAgent]].pos, targetType};
		}
	}
	return result;
}

position MoveChoice::getAgentSimpleDest(int iAgent) {
	// TODO
	return this->state.myAgents[iAgent].pos;
}

/**
	Heuristic 1
**/

MoveHeuristic_1::MoveHeuristic_1(VirtGameState& s) : MoveChoice(s) {}

target_array MoveHeuristic_1::getObjectives(int heuristic) {
	return MoveChoice::getObjectives(heuristic);
}

/**
	Heuristic 2
**/

MoveHeuristic_2::MoveHeuristic_2(VirtGameState& s) : MoveHeuristic_1(s) {}

target_array MoveHeuristic_2::getObjectives(int heuristic) {
	return MoveChoice::getObjectives(heuristic);
}

/**
	Heuristic agro
**/

MoveHeuristic_Agro::MoveHeuristic_Agro(VirtGameState& s) : MoveChoice(s) {}

target_array MoveHeuristic_Agro::getObjectives(int heuristic) {
	return MoveChoice::getObjectives(heuristic);
}